/** @file patest_sine.c
 * 2	        @brief Play a sine wave for several seconds.
 * 3	        @author Ross Bencina <rossb@audiomulch.com>
 * 4	    @author Phil Burk <philburk@softsynth.com>
 * 5	*/
/*
* $Id$
*
* This program uses the PortAudio Portable Audio Library.
* For more information see: http://www.portaudio.com/
* Copyright (c) 1999-2000 Ross Bencina and Phil Burk
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * Any person wishing to distribute modifications to the Software is
 * requested to send the modifications to the original developer so that
 * they can be incorporated into the canonical version.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

/* The program was changed by Arnold Willemer 2009 to C++ for the book
 * "Coding For Fun C++" and translated to German.
 */

#include <iostream>
using namespace std;

#include <math.h>
#include "portaudio.h"

#define NUM_SECONDS   (5)
#define SAMPLE_RATE   (44100)
#define FRAMES_PER_BUFFER  (64)

const double PI =  3.14159;

const int MAXWELLE=256;

class tSoundData {
public:
    float Welle[MAXWELLE];
    int LinksIndex;
    int RechtsIndex;
};

static int sinusCallback(const void *inputBuffer,
                         void *outputBuffer,
                         unsigned long framesPerBuffer,
                         const PaStreamCallbackTimeInfo* TI,
                         PaStreamCallbackFlags statusFlags,
                         void *userData )
{
    tSoundData *SoundData = (tSoundData*)userData;
    float *out = (float*)outputBuffer;
   
    for (unsigned int i=0; i<framesPerBuffer; i++ ) {
        *out++ = SoundData->Welle[SoundData->LinksIndex];
        *out++ = SoundData->Welle[SoundData->RechtsIndex];
        SoundData->LinksIndex += 1;
        if (SoundData->LinksIndex >= MAXWELLE) {
            SoundData->LinksIndex -= MAXWELLE;
        }
        SoundData->RechtsIndex += 3; // etwas hoeher
        if (SoundData->RechtsIndex >= MAXWELLE) {
            SoundData->RechtsIndex -= MAXWELLE;
        }
    }
    return paContinue;
}

int main()
{
    PaStreamParameters outputParameters;
    PaStream *stream;
    PaError err;
    tSoundData SoundData;

    cout << "PortAudio Test: Sinuswelle. Sample Rate = " 
         << SAMPLE_RATE << " Puffergroesse: "
         <<  FRAMES_PER_BUFFER;
 
    // Legt im Wellenspeicher eine Sinuswelle ab
    for (int i=0; i<MAXWELLE; i++ ) {
        SoundData.Welle[i] = sin((double(i)/MAXWELLE)*PI*2.0);
    }
    SoundData.LinksIndex = SoundData.RechtsIndex = 0;
	   
    err = Pa_Initialize();
    if( err != paNoError ) goto error;

    outputParameters.device = Pa_GetDefaultOutputDevice(); /* default output device */
    outputParameters.channelCount = 2;       /* stereo output */
    outputParameters.sampleFormat = paFloat32; /* 32 bit floating point output */
    outputParameters.suggestedLatency = Pa_GetDeviceInfo( outputParameters.device )->defaultLowOutputLatency;
    outputParameters.hostApiSpecificStreamInfo = NULL;

    err = Pa_OpenStream(
              &stream,
              NULL, /* no input */
              &outputParameters,
              SAMPLE_RATE,
              FRAMES_PER_BUFFER,
              paClipOff,      /* we won't output out of range samples so don't bother clipping them */
              sinusCallback,
              &SoundData );
    if( err != paNoError ) goto error;

    err = Pa_StartStream( stream );
    if( err != paNoError ) goto error;

    cout << "Play for " << NUM_SECONDS << " seconds." 
    << endl;
    Pa_Sleep( NUM_SECONDS * 1000 );

    err = Pa_StopStream( stream );
    if( err != paNoError ) goto error;

    err = Pa_CloseStream( stream );
    if( err != paNoError ) goto error;

    Pa_Terminate();
    printf("Test finished.\n");
   
    return err;
error:
    Pa_Terminate();
    fprintf( stderr, "An error occured while using the portaudio stream\n" );
    fprintf( stderr, "Error number: %d\n", err );
    fprintf( stderr, "Error message: %s\n", Pa_GetErrorText( err ) );
    return err;
}
